import Checkbox from '@mui/material/Checkbox';
import Paper from '@mui/material/Paper';
import Table from '@mui/material/Table';
import TableBody from '@mui/material/TableBody';
import TableCell from '@mui/material/TableCell';
import TableContainer from '@mui/material/TableContainer';
import TableHead from '@mui/material/TableHead';
import TableRow from '@mui/material/TableRow';
import React, { useEffect, useState } from 'react';

interface AppShellProps {
    children: React.ReactNode;
    title: string;
    renderers?: {
        [field: string]: (value: any, record: any) => React.ReactNode;
    };
    action: boolean,
    showSelection: boolean,
    dataset:array,
    columns:array,
}
export default function TableView({ columns = [], dataset = [], showSelection = false, renderers = {},action = false }: AppShellProps) {
    const [tableColumn, setTableColumns] = useState([]);

    const [tableData, setTableData] = useState([]);

    const [selectedRecords, setSelectedRecords] = useState([]);

    const canAddAction = action ?? false;

    const onSelectAllClick = (e) => {
        if (e.target.checked) {
            setSelectedRecords(tableData.map((record) => record.id));
        } else {
            setSelectedRecords([]);
        }
    };

    const onSelectClick = (e, id) => {
        if (e.target.checked) {
            setSelectedRecords([...selectedRecords, id]);
        } else {
            setSelectedRecords(selectedRecords.filter((localID) => localID != id));
        }
    };

    
    useEffect(() => {

        const newColumns = [...columns];

        if (canAddAction) {
            const actionColExists = newColumns.some((heading) => heading.field === 'action');
            if (!actionColExists) {
                newColumns.push({ label: 'Action', field: 'action' });
            }
        }

        setTableColumns(newColumns);
        setTableData(dataset);
    }, [columns, dataset,canAddAction]);

    return (
        <TableContainer component={Paper}>
            
            <Table sx={{ minWidth: 650 }} aria-label="simple table">
                <TableHead>
                    <TableRow>
                        {showSelection && (
                            <TableCell padding="checkbox">
                                <Checkbox
                                    color="primary"
                                    indeterminate={selectedRecords.length > 0 && selectedRecords.length < dataset.length}
                                    checked={dataset.length > 0 && selectedRecords.length === dataset.length}
                                    onChange={onSelectAllClick}
                                />
                            </TableCell>
                        )}
                        {tableColumn.map((column) => (
                            <TableCell>{column.label}</TableCell>
                        ))}
                    </TableRow>
                </TableHead>
                <TableBody>
                    {tableData.map((record) => (
                        <>
                            <TableRow key={record.id}>
                                <>
                                    {showSelection && (
                                        <TableCell padding="checkbox">
                                            <Checkbox
                                                color="primary"
                                                checked={selectedRecords.includes(record.id)}
                                                onChange={(e) => onSelectClick(e, record.id)}
                                            />
                                        </TableCell>
                                    )}
                                    {tableColumn.map((column, index) => {
                                        const renderer = renderers[column.field];
                                        const value = record[column.field];
                                        return (
                                            <TableCell key={index}>
                                                {renderer ? renderer(value, record) : <span dangerouslySetInnerHTML={{ __html: value ?? '-' }} />}
                                            </TableCell>
                                        );
                                    })}
                                </>
                            </TableRow>
                        </>
                    ))}
                </TableBody>
            </Table>
        </TableContainer>
    );
}
